using System;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;

namespace adorner
{

    public enum AdornerPosition
    {
        TopLeft = 0,
        TopCenter = 1,
        TopRight = 2,
        MiddleLeft = 3,
        MiddleCenter = 4,
        MiddleRight = 5,
        BottomLeft = 6,
        BottomCenter = 7,
        BottomRight = 8
    }

    class ResizingAdorner : Adorner
    {
        Thumb[] resizeThumb = new Thumb[9]; //resize handles

        VisualCollection visualChildren;
        Border adornedElement;

        Brush prevThumbBackground;
        ToolTip sizeTip = new ToolTip();


        public ResizingAdorner(Border AdornedElement)
            : base(AdornedElement)
        {
            visualChildren = new VisualCollection(this);
            adornedElement = AdornedElement;

            foreach (AdornerPosition position in Enum.GetValues(typeof(AdornerPosition)))
            {
                switch (position)
                {
                    case AdornerPosition.TopLeft:
                        BuildThumb(Cursors.SizeNWSE, position);
                        break;
                    case AdornerPosition.TopCenter:
                        BuildThumb(Cursors.SizeNS, position);
                        break;
                    case AdornerPosition.TopRight:
                        BuildThumb(Cursors.SizeNESW, position);
                        break;
                    case AdornerPosition.MiddleLeft:
                        BuildThumb(Cursors.SizeWE, position);
                        break;
                        /*
                    case AdornerPosition.MiddleCenter:
                        
                        break;
                         */
                    case AdornerPosition.MiddleRight:
                        BuildThumb(Cursors.SizeWE, position);
                        break;
                    case AdornerPosition.BottomLeft:
                        BuildThumb(Cursors.SizeNESW, position);
                        break;
                    case AdornerPosition.BottomCenter:
                        BuildThumb(Cursors.SizeNS, position);
                        break;
                    case AdornerPosition.BottomRight:
                        BuildThumb(Cursors.SizeNWSE, position);
                        break;
                }
                
            }
        }

        void BuildThumb( Cursor cursor, AdornerPosition position)
        {

            Thumb thumb = new Thumb();
            resizeThumb[(int)position] = thumb;
            thumb.Cursor = cursor;

            thumb.Tag = position; //hint for the thumb

            // resize handper for the thumb
            thumb.DragDelta += new DragDeltaEventHandler(HandleResize);
            thumb.DragStarted += onDragStarted;
            thumb.DragCompleted += onDragCompleted;

            thumb.ToolTip = sizeTip;
            thumb.Style = (Style)this.FindResource("ThumbStyle");
            SetTipContent();

            visualChildren.Add(thumb);
        }

        void Adjust(string direction, DragDeltaEventArgs args)
        {
            Double x = args.HorizontalChange;
            Double y = args.VerticalChange;
            Canvas parent = (Canvas)VisualTreeHelper.GetParent(AdornedElement);

            switch (direction)
            {
                case "top":
                    //add change and actual value
                    y += (Double)adornedElement.GetValue(Canvas.TopProperty);
                    //und dann validieren
                    if (y < 0)
                        { break; };
                    if (y > parent.ActualHeight)
                    { break; };

                    //Jetzt den oberen Rand und die Hhe mit den validierten Daten anpassen
                    adornedElement.SetValue(Canvas.TopProperty, (double)y);
                    if (args.VerticalChange < adornedElement.ActualHeight)
                    {
                        adornedElement.Height = adornedElement.ActualHeight - args.VerticalChange;
                    };
                    break;
                case "left":
                    x += (Double)adornedElement.GetValue(Canvas.LeftProperty);
                    if (x < 0)
                        { break; };
                    if (x > parent.ActualWidth)
                        { break; };
                    adornedElement.SetValue(Canvas.LeftProperty, (double)x);
                    if (args.HorizontalChange < adornedElement.ActualWidth - 0)
                    {
                        adornedElement.Width = adornedElement.ActualWidth - args.HorizontalChange;
                    };
                    break;
                case "right":
                    x += (Double)adornedElement.ActualWidth;
                    if (x < 0)
                        { break; };
                    if (x > parent.ActualWidth - (Double)adornedElement.GetValue(Canvas.LeftProperty))
                        { break; };
                    adornedElement.SetValue(WidthProperty, (double)x);
                    break;
                case "bottom":
                    y += (Double)adornedElement.ActualHeight;
                    if (y < 0)
                        { break; };
                    if (y > parent.ActualHeight - (Double)adornedElement.GetValue(Canvas.TopProperty))
                        { break; };
                    adornedElement.SetValue(HeightProperty, (double)y);
                    break;
                default:
                    break;
            }
        }

        void HandleResize(object sender, DragDeltaEventArgs args)
        {
            Double x = args.HorizontalChange;
            Double y = args.VerticalChange;
            Thumb thumb = (Thumb)sender;
            //Is the movement valid
            Canvas parent = (Canvas)VisualTreeHelper.GetParent(AdornedElement);
            AdornerPosition direction = (AdornerPosition)thumb.Tag;
            switch (direction)
            {
                case AdornerPosition.TopLeft:
                    Adjust("top", args);
                    Adjust("left", args);
                    break;
                case AdornerPosition.TopCenter:
                    Adjust("top", args);
                    break;
                case AdornerPosition.TopRight:
                    Adjust("top", args);
                    Adjust("right", args);
                    break;

                case AdornerPosition.MiddleLeft:
                    Adjust("left", args);
                    break;
                case AdornerPosition.MiddleRight:
                    Adjust("right", args);
                    break;

                case AdornerPosition.BottomLeft:
                    Adjust("bottom", args);
                    Adjust("left", args);
                    break;
                case AdornerPosition.BottomCenter:
                    Adjust("bottom", args);
                    break;
                case AdornerPosition.BottomRight:
                    Adjust("bottom", args);
                    Adjust("right", args);
                    break;
                default:
                    break;
            }
            SetTipContent();


        }

        protected void SetTipContent()
        {
            sizeTip.Content = "Width: " + adornedElement.ActualWidth + "\r\nHeight: " + adornedElement.ActualHeight;
        }

        protected override Size ArrangeOverride(Size finalSize)
        {
            Rect rect = new Rect(finalSize);
            Size size = new Size(10,10);

            foreach (AdornerPosition position in Enum.GetValues(typeof(AdornerPosition)))
            {
                Thumb thumb = resizeThumb[(int)position];
                if (thumb == null)
                    continue;

                AdornerPosition direction = (AdornerPosition)thumb.Tag;
                switch (direction)
                {
                    case AdornerPosition.TopLeft:
                        thumb.Arrange(new Rect(new Point(rect.Left - 5, rect.Top - 5), size));
                        break;
                    case AdornerPosition.TopCenter:
                        thumb.Arrange(new Rect(new Point((rect.Right / 2) - 5, rect.Top - 5), size));
                        break;

                    case AdornerPosition.TopRight:
                        thumb.Arrange(new Rect(new Point(rect.Right - 5, rect.Top - 5), size));
                        break;

                    case AdornerPosition.MiddleLeft:
                        thumb.Arrange(new Rect(new Point(rect.Left - 5, rect.Bottom / 2 - 5), size));
                        break;
                    case AdornerPosition.MiddleRight:
                        thumb.Arrange(new Rect(new Point(rect.Right - 5, rect.Bottom / 2 - 5), size));
                        break;

                    case AdornerPosition.BottomLeft:
                        thumb.Arrange(new Rect(new Point(rect.Left - 5, rect.Bottom - 5), size));
                        break;
                    case AdornerPosition.BottomCenter:
                        thumb.Arrange(new Rect(new Point(rect.Right / 2 - 5, rect.Bottom - 5), size));
                        break;
                    case AdornerPosition.BottomRight:
                        thumb.Arrange(new Rect(new Point(rect.Right - 5, rect.Bottom - 5), size));
                        break;
                    default:
                        break;
                }

            }

            return base.ArrangeOverride(finalSize);
        }


        void onDragStarted(object sender, DragStartedEventArgs e)
        {
            Thumb thumb = (Thumb)sender;
            prevThumbBackground = thumb.Background;
            thumb.Background = Brushes.Blue;
        }

        void onDragCompleted(object sender, DragCompletedEventArgs e)
        {
            Thumb thumb = (Thumb)sender;
            thumb.Background = prevThumbBackground;
        }
 

        protected override int VisualChildrenCount { get { return visualChildren.Count; } }
        protected override Visual GetVisualChild(int index) { return visualChildren[index]; }
    }
}
